<?php
/**
 * @package         Conditions
 * @version         23.9.3039
 * 
 * @author          Peter van Westen <info@regularlabs.com>
 * @link            https://regularlabs.com
 * @copyright       Copyright © 2023 Regular Labs All Rights Reserved
 * @license         GNU General Public License version 2 or later
 */

namespace RegularLabs\Component\Conditions\Administrator\Form\Field;

defined('_JEXEC') or die;

use Joomla\CMS\Language\Text as JText;
use Joomla\CMS\Object\CMSObject as JCMSObject;
use RegularLabs\Component\Conditions\Administrator\Helper\Helper;
use RegularLabs\Component\Conditions\Administrator\Helper\Summary;
use RegularLabs\Component\Conditions\Administrator\Model\ItemModel;
use RegularLabs\Library\Document as RL_Document;
use RegularLabs\Library\Form\FormField as RL_FormField;
use RegularLabs\Library\Language as RL_Language;
use RegularLabs\Library\ShowOn as RL_ShowOn;

class ConditionField extends RL_FormField
{
    private $enabled_types;
    private $extension;
    private $item_id;
    private $name_column;
    private $table;
    private $text_no_conditions;

    public function getCondition()
    {
        return (new ItemModel)->getConditionByExtensionItem(
            $this->extension,
            $this->item_id,
            false,
            $this->enabled_types
        );
    }

    protected function getButtonUrl($view, $task = 'edit', $layout = 'modal')
    {
        return 'index.php?option=com_conditions'
            . '&view=' . $view
            . ($task ? '&task=' . $view . '.' . $task : '')
            . '&extension=' . $this->extension
            . '&item_id=' . $this->item_id
            . '&table=' . $this->table
            . '&name_column=' . $this->name_column
            . '&enabled_types=' . $this->enabled_types
            . '&message=' . $this->text_no_conditions
            . '&layout=' . $layout
            . '&tmpl=component';
    }

    protected function getInput()
    {
        $this->extension          = $this->get('extension', $this->app->input->get('option', ''));
        $this->item_id            = $this->get('item_id', $this->app->input->getInt('id'));
        $this->table              = $this->get('table', $this->app->input->getInt('table'));
        $this->name_column        = $this->get('name_column', $this->app->input->getInt('name_column'));
        $this->text_no_conditions = $this->get('text_no_conditions', $this->app->input->get('message', ''));
        $this->enabled_types      = $this->get('enable', '');
        $this->enabled_types      = str_replace(' ', '', $this->enabled_types);

        if ( ! $this->extension)
        {
            return '';
        }

        if ( ! $this->item_id)
        {
            return $this->getHtmlForNoItemId();
        }

        RL_Document::script('regularlabs.regular');
        RL_Document::script('conditions.script');
        RL_Language::load($this->extension);

        $condition = $this->getCondition();

        if ($condition && $condition->published !== 1)
        {
            $condition = null;
        }

        $has_conditions = ! empty($condition);

        $html   = [];
        $html[] = '<input type="hidden" name="' . $this->getName('has_conditions') . '" id="' . $this->getId('', 'has_conditions') . '" value="' . (int) $has_conditions . '">';
        $html[] = '<input type="hidden" name="' . $this->getName('condition_id') . '" id="' . $this->getId('', 'condition_id') . '" value="' . ($condition->id ?? '') . '">';
        $html[] = '<input type="hidden" name="' . $this->getName('condition_alias') . '" id="' . $this->getId('', 'condition_alias') . '" value="' . ($condition->alias ?? '') . '">';
        $html[] = '<input type="hidden" name="' . $this->getName('condition_name') . '" id="' . $this->getId('', 'condition_name') . '" value="' . ($condition->name ?? '') . '">';
        $html[] = '<div id="rules_summary" class="position-relative">';
        $html[] = '<div id="rules_summary_message" class="alert alert-warning' . ($has_conditions ? ' hidden' : '') . '">'
            . '<span class="icon-info-circle text-info" aria-hidden="true"></span> '
            . JText::_('CON_MESSAGE_NO_CONDITION_SELECTED')
            . ($this->text_no_conditions ? '<br><br>' . JText::_($this->text_no_conditions) : '')
            . '</div>';
        $html[] = $this->getButtons();
        $html[] = '<div id="rules_summary_content" class="mt-4">';
        $html[] = Summary::render($condition, $this->extension, $this->text_no_conditions, $this->enable);
        $html[] = '</div >';
        $html[] = '</div >';

        return implode('', $html);
    }

    protected function getLabel()
    {
        return '';
    }

    private function addSaveButtons(&$options)
    {
        $modal     = 'this.closest(\'.modal-content\')';
        $iframe    = $modal . '.querySelector(\'.modal-body > iframe\')';
        $hide_self = 'this.classList.add(\'hidden\');';
        $apply     = $hide_self . $modal . ' && ' . $iframe . ' && ' . $iframe . '.contentWindow.Joomla.submitbutton(\'item.apply\')';
        $save      = $hide_self . $modal . ' && ' . $iframe . ' && ' . $iframe . '.contentWindow.Joomla.submitbutton(\'item.save\')';

        $options = array_merge($options, [
                'keyboard'         => false,
                'backdrop'         => 'static',
                'confirmText'      => JText::_('JAPPLY'),
                'confirmCallback'  => $apply,
                'confirmClass'     => 'btn btn-success hidden conditions-button',
                'confirmIcon'      => 'save',
                'confirm2Text'     => JText::_('JSAVE'),
                'confirm2Callback' => $save,
                'confirm2Class'    => 'btn btn-success hidden conditions-button',
                'confirm2Icon'     => 'save',
            ]
        );
    }

    private function getButton($name, $text, $link, $icon, $class = 'primary', $options = [])
    {
        $button          = new JCMSObject;
        $button->name    = $this->id . '_' . $name;
        $button->text    = JText::_($text);
        $button->icon    = $icon;
        $button->class   = 'btn-' . $class . ' mb-1';
        $button->options = $options;

        if ($link)
        {
            $button->link = $link;
        }

        return $button;
    }

    private function getButtonCreate()
    {
        $options = [];
        $this->addSaveButtons($options);

        return $this->renderButtonAndModal(
            'create',
            'CON_BUTTON_CREATE',
            $this->getButtonUrl('item', 'modaledit', 'modal_edit'),
            'file-add',
            'success',
            $options
        );
    }

    private function getButtonDelete()
    {
        return $this->renderButtonAndModal(
            'delete',
            'CON_BUTTON_REMOVE',
            $this->getButtonUrl('item', 'remove_mapping'),
            'times',
            'danger',
            [
                'height'     => '400px',
                'width'      => '400px',
                'bodyHeight' => '400px',
                'modalWidth' => '400px',
            ]
        );
    }

    private function getButtonEdit()
    {
        $options = [];
        $this->addSaveButtons($options);

        return $this->renderButtonAndModal(
            'edit',
            'CON_BUTTON_EDIT',
            $this->getButtonUrl('item', 'modaledit'),
            'edit',
            'warning',
            $options
        );
    }

    private function getButtonSelect()
    {
        if ( ! Helper::thereAreConditions())
        {
            return '';
        }

        return $this->renderButton(
            'select',
            'CON_BUTTON_SELECT',
            $this->getButtonUrl('items', ''),
            'hand-pointer'
        );
    }

    private function getButtons()
    {
        $html = [
            RL_ShowOn::open('has_conditions:1', $this->formControl),
            $this->getButtonSelect(),
            $this->getButtonEdit(),
            $this->getButtonDelete(),
            RL_ShowOn::close(),

            RL_ShowOn::open('has_conditions:0', $this->formControl),
            $this->getButtonSelect(),
            $this->getButtonCreate(),
            RL_ShowOn::close(),

            $this->getModalSelect(),
        ];

        return implode('', $html);
    }

    private function getHtmlForNoItemId()
    {
        $item_name = Helper::getExtensionItemString($this->extension);

        return '<div class="alert alert-warning">'
            . '<span class="icon-warning"></span> '
            . JText::_('CON_MESSAGE_SAVE_ITEM_BEFORE_CREATING')
            . '<br><br>'
            . JText::sprintf('CON_MESSAGE_SAVE_AS_UNPUBLISHED', $item_name)
            . '<br><br>'
            . $this->getSaveButton(true)
            . $this->getSaveButton(false)
            . '</div>';
    }

    private function getModalButton($name, $text, $link, $icon, $class = 'primary', $options = [])
    {
        $button = $this->getButton($name, $text, $link, $icon, $class, $options);

        $button->modal   = true;
        $button->options = array_merge([
            'height'     => '400px',
            'width'      => '800px',
            'bodyHeight' => '70',
            'modalWidth' => '80',
        ], $button->options);

        return $button;
    }

    private function getModalSelect()
    {
        return $this->renderModal(
            'select',
            'CON_BUTTON_SELECT',
            $this->getButtonUrl('items', ''),
            'hand-pointer'
        );
    }

    private function getSaveButton($published = true)
    {
        $button = $this->getButton(
            $published ? 'save-published' : 'save-unpublished',
            $published ? 'CON_SAVE_AS_PUBLISHED' : 'CON_SAVE_AS_UNPUBLISHED',
            null,
            'save',
            $published ? 'warning' : 'success'
        );

        $button->onclick = '$("select#jform_state").val(' . ($published ? 1 : 0) . ');'
            . '$("select#jform_published").val(' . ($published ? 1 : 0) . ');'
            . '$("joomla-toolbar-button#toolbar-apply > button").click();';

        return $this->getRenderer('regularlabs.buttons.button')
            ->addIncludePath(JPATH_SITE . '/libraries/regularlabs/layouts')
            ->render($button);
    }

    private function renderButton($name, $text, $link, $icon, $class = 'primary', $options = [])
    {
        $button = $this->getModalButton($name, $text, $link, $icon, $class, $options);

        return $this->getRenderer('regularlabs.buttons.button')
            ->addIncludePath(JPATH_SITE . '/libraries/regularlabs/layouts')
            ->render($button);
    }

    private function renderButtonAndModal($name, $text, $link, $icon, $class = 'primary', $options = [])
    {
        return $this->renderButton($name, $text, $link, $icon, $class, $options)
            . $this->renderModal($name, $text, $link, $icon, $class, $options);
    }

    private function renderModal($name, $text, $link, $icon, $class = 'primary', $options = [])
    {
        $button = $this->getModalButton($name, $text, $link, $icon, $class, $options);

        return $this->getRenderer('regularlabs.buttons.modal')
            ->addIncludePath(JPATH_SITE . '/libraries/regularlabs/layouts')
            ->render($button);
    }
}
