<?php
/**
 * @package         Conditions
 * @version         23.9.3039
 * 
 * @author          Peter van Westen <info@regularlabs.com>
 * @link            https://regularlabs.com
 * @copyright       Copyright © 2023 Regular Labs All Rights Reserved
 * @license         GNU General Public License version 2 or later
 */

namespace RegularLabs\Component\Conditions\Administrator\Model;

use Joomla\CMS\Factory as JFactory;
use Joomla\CMS\Language\Text as JText;
use Joomla\CMS\MVC\Model\ListModel;
use RegularLabs\Library\Cache as RL_Cache;
use RegularLabs\Library\Parameters as RL_Parameters;

defined('_JEXEC') or die;

class ItemsModel extends ListModel
{
    protected $config;

    /**
     * @var     string    The prefix to use with controller messages.
     */
    protected $text_prefix = 'RL';

    /**
     * Constructor.
     *
     * @param array    An optional associative array of configuration settings.
     *
     * @see        JController
     */
    public function __construct($config = [])
    {
        if (empty($config['filter_fields']))
        {
            $config['filter_fields'] = [
                'alias', 'a.alias',
                'category', 'a.category',
                'color', 'a.color',
                'description', 'a.description',
                'id', 'a.id',
                'name', 'a.name',
                'published', 'a.published',
            ];
        }

        parent::__construct($config);

        $is_modal = JFactory::getApplication()->input->getString('layout') === 'modal';

        if ($is_modal)
        {
            $this->context .= '_modal';
        }

        $this->config = RL_Parameters::getComponent('conditions');
    }

    /**
     * Duplicate Method
     * Duplicate all items specified by array id
     */
    public function duplicate($ids, $model)
    {
        foreach ($ids as $id)
        {
            $model->duplicate($id);
        }

        $msg = JText::sprintf('%d items duplicated', count($ids));
        JFactory::getApplication()->enqueueMessage($msg);
    }

    public function getHasCategories()
    {
        $cache = new RL_Cache;

        if ($cache->exists())
        {
            return $cache->get();
        }

        $db = $this->getDbo();

        $query = $db->getQuery(true)
            ->select('COUNT(*)')
            ->from($db->quoteName('#__conditions'))
            ->where($db->quoteName('category') . ' != ' . $db->quote(''));

        $db->setQuery($query);

        return $cache->set($db->loadResult());
    }

//    /**
//     * @param array   $data     Data for the form.
//     * @param boolean $loadData True if the form is to load its own data (default case), false if not.
//     *
//     * @return  JForm   A Form object on success, false on failure
//     */
//    public function getForm($data = [], $loadData = true)
//    {
//        return $this->loadForm('com_conditions.items', 'items');
//    }

    public function getItems($getall = false, $aliases = [])
    {
        $cache = new RL_Cache;

        if ($cache->exists())
        {
            return $cache->get();
        }

        // Load the list items.
        if ($getall)
        {
            $db    = $this->getDbo();
            $query = $db->getQuery(true)
                // Select the required fields from the table.
                ->select('a.*')
                ->from($db->quoteName('#__conditions', 'a'));
        }
        else
        {
            $query = $this->_getListQuery();
        }

        $sub_query = $this->_db->getQuery(true)
            ->select('count(*)')
            ->from('#__conditions_map as m')
            ->where('m.condition_id = a.id');
        $query->select('(' . ((string) $sub_query) . ') as nr_of_uses');

        if ($getall)
        {
            $this->_db->setQuery($query);
            $items = $this->_db->loadObjectList('alias');
        }
        else
        {
            $items = $this->_getList($query, $this->getStart(), $this->getState('list.limit'));
        }

        return $cache->set($items);
    }

    public function hasItems()
    {
        $cache = new RL_Cache;

        if ($cache->exists())
        {
            return $cache->get();
        }

        $query = $this->_db->getQuery(true)
            ->select('count(*)')
            ->from($this->_db->quoteName('#__conditions', 'a'))
            ->where('( ' . $this->_db->quoteName('a.published') . ' IN ( 0,1,2 ) )');

        $this->_db->setQuery($query);

        return $cache->set($this->_db->loadResult() > 0);
    }

    /**
     * Build an SQL query to load the list data.
     *
     * @return    JDatabaseQuery
     */
    protected function getListQuery()
    {
        $db = $this->getDbo();

        $query = $db->getQuery(true)
            // Select the required fields from the table.
            ->select(
                $this->getState(
                    'list.select',
                    'a.*'
                )
            )
            ->from($db->quoteName('#__conditions', 'a'));

        $state = $this->getState('filter.state');

        if (is_numeric($state))
        {
            $query->where($db->quoteName('a.published') . ' = ' . ( int ) $state);
        }
        elseif ($state == '')
        {
            $query->where('( ' . $db->quoteName('a.published') . ' IN ( 0,1,2 ) )');
        }

        $category = $this->getState('filter.category');

        if ($category != '')
        {
            $query->where($db->quoteName('a.category') . ' = ' . $db->quote($category));
        }

        // Filter the list over the search string if set.
        $search = $this->getState('filter.search');

        if ( ! empty($search))
        {
            if (stripos($search, 'id:') === 0)
            {
                $query->where($db->quoteName('a.id') . ' = ' . ( int ) substr($search, 3));
            }
            else
            {
                $search = $db->quote('%' . $db->escape($search, true) . '%');
                $query->where(
                    '( ' . $db->quoteName('a.alias') . ' LIKE ' . $search .
                    ' OR ' . $db->quoteName('a.name') . ' LIKE ' . $search .
                    ' OR ' . $db->quoteName('a.description') . ' LIKE ' . $search .
                    ' OR ' . $db->quoteName('a.category') . ' LIKE ' . $search . ' )'
                );
            }
        }

        $query->select($db->quoteName('uc.name', 'editor'))
            ->join('LEFT', $db->quoteName('#__users', 'uc'), $db->quoteName('uc.id') . ' = ' . $db->quoteName('a.checked_out'));

        // Add the list ordering clause.
        $ordering = $this->getState('list.ordering', 'a.name');

        $query->order($db->quoteName($db->escape($ordering)) . ' ' . $db->escape($this->getState('list.direction', 'ASC')));

        return $query;
    }

    /**
     * Method to get a store id based on model configuration state.
     *
     * This is necessary because the model is used by the component and
     * different modules that might need different sets of data or different
     * ordering requirements.
     *
     * @param string    A prefix for the store id.
     *
     * @return    string    A store id.
     */
    protected function getStoreId($id = '', $getall = 0, $alias = [])
    {
        // Compile the store id.
        $id .= ':' . $this->getState('filter.search');
        $id .= ':' . $this->getState('filter.state');
        $id .= ':' . $getall;
        $id .= ':' . json_encode($alias);

        return parent::getStoreId($id);
    }

    /**
     * Method to auto-populate the model state.
     *
     * Note. Calling getState in this method will result in recursion.
     *
     */
    protected function populateState($ordering = null, $direction = null)
    {
        // List state information.
        parent::populateState('a.name', 'asc');
    }
}
